#define GOAL_WIDTH 400
#define SCALE_RATIO 200
#include "GameLayer.h"


using namespace cocos2d;

GameLayer::GameLayer() {
	
}

Scene* GameLayer::scene() {
	auto scene = Scene::create();	
	
	auto layer = GameLayer::create();

	scene->addChild(layer);
	

	return scene;
}

bool GameLayer::init() {
	// call to super
	if (!Layer::init())
	{
		return false;
	}

	this->initPhysics();
	
	_screenSize = Director::getInstance()->getWinSize();
	_center = Vec2(_screenSize.width * 0.5, _screenSize.height * 0.5);
	_delta = Vec2(0,0);

	_ball = GameSprite::gameSpriteWithFile("res/ball.png");
	_ball->setPosition(_center);
	this->addChild(_ball);

	_box = GameSprite::gameSpriteWithFile("res/box.png");
	_box->setPosition(_center);
	this->addChild(_box);

	_ground = GameSprite::gameSpriteWithFile("res/ground.png");
	_ground->setPosition(Vec2(_center.x, 16.0f));
	this->addChild(_ground);

	// Create a static ground body
	b2BodyDef groundBodyDef;
	groundBodyDef.position.Set(_center.x / SCALE_RATIO, 16.0f / SCALE_RATIO);
	_staticBody = _world->CreateBody(&groundBodyDef);

	b2PolygonShape groundBox;
	groundBox.SetAsBox(800.0f / 2 / SCALE_RATIO, 32.0f / 2 / SCALE_RATIO);
	
	_staticBody->CreateFixture(&groundBox, 0.0f);


	//create a dynamic body for the ball
	b2BodyDef bodyDef;
	bodyDef.type = b2_dynamicBody;
	bodyDef.position.Set(_center.x/SCALE_RATIO, _center.y / SCALE_RATIO);
	bodyDef.userData = _ball;
	_dynamicBody = _world->CreateBody(&bodyDef);

	b2CircleShape ballShape;
	ballShape.m_radius = (17.5f / SCALE_RATIO);

	b2FixtureDef fixtureDef;
	fixtureDef.shape = &ballShape;
	fixtureDef.density = 1.0f;
	fixtureDef.friction = 0.3f;
	fixtureDef.restitution = 0.6;

	_dynamicBody->CreateFixture(&fixtureDef);

	createDynamicBox(600.0f, 32.0f, 32.0f, 32.0f);




	auto listener = EventListenerTouchAllAtOnce::create();
	listener->onTouchesBegan = CC_CALLBACK_2(GameLayer::onTouchesBegan, this);
	listener->onTouchesMoved = CC_CALLBACK_2(GameLayer::onTouchesMoved, this);
	listener->onTouchesEnded = CC_CALLBACK_2(GameLayer::onTouchesEnded, this);
	_eventDispatcher->addEventListenerWithSceneGraphPriority(listener, this);
	

	

	//create main loop
	this->scheduleUpdate();
	return true;

}

void GameLayer::createDynamicBox(float xPos, float yPos, float width, float height)
{
	//create a dynamic body
	b2BodyDef bodyDef;
	bodyDef.type = b2_dynamicBody;
	bodyDef.userData = _box;
	bodyDef.position.Set(xPos / SCALE_RATIO, yPos / SCALE_RATIO);
	b2Body * box = _world->CreateBody(&bodyDef);

	b2PolygonShape boxShape;
	boxShape.SetAsBox(width / 2 / SCALE_RATIO, height / 2 / SCALE_RATIO);

	b2FixtureDef fixtureDef;
	fixtureDef.shape = &boxShape;
	fixtureDef.density = 10.0f;
	fixtureDef.friction = 0.4f;
	fixtureDef.restitution = 0.1f;

	box->CreateFixture(&fixtureDef);

}

void GameLayer::createBoxes() {

	for (int i = 0; i<4; i++) {
		createDynamicBox(300  + i*40 , 20, 10, 40);
	}
	createDynamicBox(360, 65, 75, 10);

}

void GameLayer::initPhysics()
{
	_timeStep = 1.0f / 60.0f;

	// Create a world, define gravity
	b2Vec2 gravity = b2Vec2(0.0f, -8.0f);
	_world = new b2World(gravity);
	
	
	_debugDraw = new GLESDebugDraw(SCALE_RATIO);
	_world->SetDebugDraw(_debugDraw);

	uint32 flags = 0;
	flags += b2Draw::e_shapeBit;
	//        flags += b2Draw::e_jointBit;
	//        flags += b2Draw::e_aabbBit;
	//        flags += b2Draw::e_pairBit;
	//        flags += b2Draw::e_centerOfMassBit;
	_debugDraw->SetFlags(flags);

}

void GameLayer::draw(Renderer *renderer, const Mat4& transform, uint32_t flags)
{
	Layer::draw(renderer, transform, flags);
	
	Director* director = Director::getInstance();

	GL::enableVertexAttribs(cocos2d::GL::VERTEX_ATTRIB_FLAG_POSITION);
	director->pushMatrix(MATRIX_STACK_TYPE::MATRIX_STACK_MODELVIEW);
	_world->DrawDebugData();
	director->popMatrix(MATRIX_STACK_TYPE::MATRIX_STACK_MODELVIEW);
	
}


void GameLayer::update(float dt) {
	int velocityIterations = 10;
	int positionIterations = 10;

	_world->Step(dt, velocityIterations, positionIterations);

	for (b2Body *body = _world->GetBodyList(); body != NULL; body = body->GetNext())
		if (body->GetUserData())
		{
			GameSprite *sprite = (GameSprite *)body->GetUserData();
			sprite->setPosition(ccp(body->GetPosition().x * SCALE_RATIO, body->GetPosition().y * SCALE_RATIO));
			sprite->setRotation(-1 * CC_RADIANS_TO_DEGREES(body->GetAngle()));

		}

	_world->ClearForces();


}



void GameLayer::onTouchesBegan(const std::vector<Touch*> &touches, Event* event)
{
	for (auto touch : touches) {
		if (touch != nullptr) {
			_delta = touch->getLocation();
		}
	}
}

void GameLayer::onTouchesMoved(const std::vector<Touch*> &touches, Event* event){
	for (auto touch : touches) {
		if (touch != nullptr) {
			auto tap = touch->getLocation();
		
			
		}
	}
}

void GameLayer::onTouchesEnded(const std::vector<Touch*> &touches, Event* event)
{
	for (auto touch : touches) {
		if (touch != nullptr) {
			Vec2 tap = touch->getLocation();
			_dynamicBody->ApplyForce(b2Vec2((tap.x - _delta.x)/SCALE_RATIO, (tap.y - _delta.y) / SCALE_RATIO), _dynamicBody->GetWorldCenter(), true);
		}
	}
}





GameLayer::~GameLayer()
{
}

